/* This program was burned from David Tait pic16c84 programm ,so
 * to program PIC12C508/509 you need the same hardware attached to lpt1 or lpt2.
 *
 * pp12.c : A serial mode PIC12C508/509 programmer
 *
 * Usage:
 *
 *      pp12  [ -lxhrwpc2389! ]  objfile1  [ objfile2 ]
 *
 * Where objfile1 is the program data in Intel hex format
 * and the optional objfile2 is id_data in the same format.
 * Various things, including the fuse details, can be set on the
 * command line using the switches described below.  If no fuse
 * details are given, the program asks for them before proceeding.
 * Superfluous command line arguments are silently ignored and later
 * switches take precedence, even if contradictory.
 *
 * Switches (options) are introduced by - (or /) and are encoded as
 * a sequence of one or more characters from the set {lxhrwpc238!}.
 * Switches can be given individually (like -r /2 -p) or in groups
 * (like -r2p or /rp2 or /2/r/p which are all synonyms).  Switches are
 * recognized anywhere in the command line (but don't use the DOS-like
 * pp/r/p/2  form - this should be written  pp /r/p/2;  space around
 * switch groups is important).
 *
 * There are four logical groups of switches: fuses, ports, hex-types
 * and misc.  They can be mixed freely.
 *
 * Fuses:
 *    o    l, x, i, r  -  LP, XT, internal or external RC oscillator (default LP).
 *    o    w           -  enable watchdog timer.
 *    o    m           -  enable MCLR pin.
 *    o    c           -  enable code protection.
 * Ports:
 *    o   2, 3         -  hardware on LPT2 or LPT3 (default is LPT1).
 * Hex-types:
 *    o   8            -  objfiles are Intel Hex-8, i.e. INHX8M
 *                        (default is Intel Hex-16, i.e. INHX16)
 * Processor-type:
 *    o	  9	       -  12c509 (1024 Program Data Words)
 *		          (default is 12c508 (512 Program Data Words)
 * Misc:
 *    o   !            -  start programming immediately (default is
 *                        to wait for a key press)
 *
 * E.g., -rp28 means set the fuses to use the RC oscillator, no watchdog,
 * use power-up timer, and don't protect code; assume the hardware is on
 * LPT2; and the objfiles are in INHX8M format.
 *
 *
 * Notes/restrictions/bugs:
 *
 *      o  This program needs external hardware to be attached
 *         to a parallel port (see pp.asc for an ASCII schematic).
 *      o  Don't use from Windows; another program could grab the LPT port.
 *      o  It's highly recomended uninstall  QEMM or emm managers.
 *      o  Run pp, pp -2 or pp -3 to initialise the LPT port.
 *      o  No support for imbedded fuse and ID information in hex file.
 *      o  EEPROM data is assumed to be word-sized even if INHX8M is used.
 *      o  This is NOT a production quality programmer. (No surprise!).
 *
 *
 * Revision history:

 *
 * This version (V-1.0b) is a beta release.  If you receive this program
 * consider yourself a beta tester (of course, you are under no
 * obligation).  I will remove beta status when 10 (arbitrary
 * I know) testers tell me they have used it to successfully program a
 * PIC using the hardware in pp.asc.
 *
 *
 *
 *
 * Acknowledgements:
 *
 * The programming algorithms used are taken from the data sheet
 * DS... (C) 1996 Microchip Technology Incorporated.  Thanks
 * to David Tait for pic16c84 prototype program.


 * THIS PROGRAM IS PROVIDED AS IS AND WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <conio.h>

/* #define U7407 */   /* define U7407 if H/W uses non-inverting buffers */

#define IN      64
#define VPP     8
#define VDD     4
#define CLK     2
#define OUT     1

#define LDCONF  0
#define LDPROG  2
#define RDPROG  4
#define INCADD  6
#define BEGPRG  8
#define ENDPRG	14
#define LDDATA  3
#define RDDATA  5

#define VPPDLY  100    /* these are in milliseconds */
#define PRGDLY  12

#define TSET    0      /* the units for these are CPU dependent */
#define THLD    0      /* I doubt if they are necessary even    */
#define TDLY    2      /* on the fastest PC, but ...            */

#define PROGRAM 0
#define DATA    1
/* #define PSIZE   512   */
#define IDSIZE  4

#define CP      8
#define MCLRE   16
#define WDTE    4
#define XRC     3
#define IRC     2
#define XT      1
#define LP      0

#define INHX16  16
#define INHX8M  8

#ifdef U7407
#define inbit           (inportb(s_reg)&IN)
#define vppon           (d_bits |= VPP)
#define vppoff          (d_bits &= ~VPP)
#define vddon           (d_bits |= VDD)
#define vddoff          (d_bits &= ~VDD)
#define clkhi           (d_bits |= CLK)
#define clklo           (d_bits &= ~CLK)
#define outhi           (d_bits |= OUT)
#define outlo           (d_bits &= ~OUT)
#else
#define inbit           (~inportb(s_reg)&IN)
#define vppon           (d_bits &= ~VPP)
#define vppoff          (d_bits |= VPP)
#define vddon           (d_bits &= ~VDD)
#define vddoff          (d_bits |= VDD)
#define clkhi           (d_bits &= ~CLK)
#define clklo           (d_bits |= CLK)
#define outhi           (d_bits &= ~OUT)
#define outlo           (d_bits |= OUT)
#endif 			/* U7407 */
#define assert          (outportb(d_reg,d_bits))

int PSIZE = 512;
int progbuf[1024];
int idbuf[IDSIZE];
int fuses = CP+XRC;

int d_bits = 0;
int d_reg;
int s_reg;
int check;
int iport = 0;
int hextype = INHX16;
int set_fuses = 0;
int wait = 1;

char *progname = "PIC12c508/509 Programmer";
char *version = "Version 0.0b";
char *copyright = "Copyright (C) 1994 David Tait; (C) 1997 Kustarev Pavel.";
char *firm = "This program is created in Laboratory of Microprocessor Technologies.";
char *adress = "Russia, St-Petesburg, tel. 812-238-85-24 ";
char *email = "e_mail: kustarev@d1.ifmo.ru   URL: http:\\lmt.ifmo.ru";



void idle_mode(void)
{
   vppoff, clklo, outlo, assert;
   delay(VPPDLY);
   vddoff, assert;
}

void prog_mode(void)
{
   vppoff, vddon, clklo, outlo, assert;
   delay(VPPDLY);
   vppon, assert;
}

void quit(char* s)
{
   idle_mode();
   fprintf(stderr,"pp: %s\n",s);
   sound(1000); delay(1000); nosound(); /* error sound signaling ! */
   exit(1);
}



void usage(void)
{
   printf("\n%s  %s \n%s\n\n%s\n%s\n\n",progname,version,copyright,firm,adress);
   printf("Usage: pp  [ -lxirwmc2389! ]  objfile1  [ objfile2 ]\n\n");
   printf("       objfile1 = Program data, objfile2 = ID data\n\n");
   printf("       Fuses:    l = LP,   x = XT,    i = internal RC,  r = external RC\n");
   printf("                 w = WDTE, m = MCLRE, c = !CP (i.e. protect)\n");
   printf("       Port:     2 = LPT2, 3 = LPT3\n");
   printf("       Hex:      8 = INHX8M\n");
   printf("       Processor:9 =12c509\n");
   printf("       Misc:     ! = no wait\n\n");
   printf("       Defaults: LP, !WDTE, !MCLRE, CP, LPT1, INHX16, wait, 12c508\n\n");
   printf("Bug reports to %s\n",email);
   idle_mode();
   exit(1);
}

void test_hw(void)
{
    int b;

    vddon, outhi, assert;   /* better have VDD on in case PIC is there */
    delay(VPPDLY);
    b = inbit;
    outlo, assert;
    if ( b != IN || inbit != 0 ) {
       fprintf(stderr,"pp: Using LPT at 0x%04X\n",d_reg);
       quit("Hardware fault.  Check power, connections and port used.");
    }
    idle_mode();
}

void setup(void)
{
   int i;
   /*, b;*/

   vppoff, vddoff, clklo, outlo, assert;
   d_reg = peek(0,0x408+iport);  /* find the base address of LPT port */
   s_reg = d_reg+1;

   switch ( d_reg ) {  /* I don't want to blow anything up, so check port */
       case 0x3BC:
       case 0x378:
       case 0x278: break;
       default:  fprintf(stderr,"pp: LPT at 0x%04X\n",d_reg);
		 quit("Unlikely LPT address");
   }

   test_hw();

   for ( i=0; i<PSIZE; ++i )
      progbuf[i] = 0x0FFF;
   for ( i=0; i<IDSIZE; ++i )
      idbuf[i] = 0xFF;
}

void tiny_delay(int ticks)
{
   while ( ticks-- )
      ;
}

void clock_out(int bit)
{
   bit? outhi: outlo, clkhi, assert;
   tiny_delay(TSET);
   clklo, assert;
   tiny_delay(THLD);
   outlo, assert;
}

int clock_in(void)
{
   int b;

   outhi, clkhi, assert;
   tiny_delay(TSET);
   clklo, assert;
   b = inbit? 1: 0;
   tiny_delay(THLD);
   return b;
}

void out_word(int w)
{
   int b;

   clock_out(0);
   for ( b=0; b<14; ++b )
     clock_out(w&(1<<b));
   clock_out(0);
}

int in_word(void)
{
   int b, w;

   (void) clock_in();
   for ( w=0, b=0; b<14; ++b )
     w += clock_in()<<b;
   (void) clock_in();
   return w;
}

void command(int cmd)
{
   int b;

   outlo, assert;
   tiny_delay(TDLY);
   for ( b=0; b<6; ++b )
      clock_out(cmd&(1<<b));
   outhi, assert;
   tiny_delay(TDLY);
}

void get_option(char* s)
{
   char *sp;

   for ( sp=s; *sp; ++sp )
      switch ( *sp ) {
	 case 'l':
	 case 'L': fuses = (fuses&0x1C) + LP; set_fuses = 1; break;
	 case 'x':
	 case 'X': fuses = (fuses&0x1C) + XT; set_fuses = 1; break;
	 case 'i':
	 case 'I': fuses = (fuses&0x1C) + IRC; set_fuses = 1; break;
	 case 'r':
	 case 'R': fuses = (fuses&0x1C) + XRC; set_fuses = 1; break;
	 case 'w':
	 case 'W': fuses |= WDTE; set_fuses = 1; break;
	 case 'm':
	 case 'M': fuses |= MCLRE; set_fuses = 1; break;
	 case 'c':
	 case 'C': fuses &= ~CP; set_fuses = 1; break;
	 case '2': iport = 2; break;
	 case '3': iport = 4; break;
	 case '8': hextype = INHX8M; break;
	 case '9': PSIZE = 1024; break;
	 case '!': wait = 0; break;
	 case '-':
	 case '/': break;
	 default: usage();
      }
}

#define yes     (c=getche(), c == 'y' || c == 'Y')

void ask_fuses(void)
{
   int c, osc, wdte, pwrte, cp;

   do {
      printf("\nOscillator type:\n 0)-LP  1)-XT  2)-intRC  3)-extRC ? ");
      osc = getche() - '0';
   } while ( osc < 0 || osc > 3 );

   printf("\nEnable watchdog timer (y/n) ? ");
   wdte = yes? WDTE: 0;
   printf("\nEnable MCLR pin (y/n) ? ");
   pwrte = yes? MCLRE: 0;
   printf("\n         Protect code (y/n) ? ");
   cp = yes?  0: CP;
   printf("\n");
   fuses = osc + wdte + pwrte + cp;
}
/*-----------------------------------------------------------------------*/



/*---------------- mashine independent tiny delay  ------------*/
void dly100us(void)
{
	int w;

	outportb(0x61,0x40);    /* {disable speaker} */

	outportb(0x43,0x96);    /* set up timer mode  */

	outportb(0x42,116);     /* set up timer COUNTER  */

	outportb (0x61,0x41);	/* start   timer   */

	do
	w=inportb(0x61) & 0x20;  /*check back front from timer out 2 */
	while (w!=0);

	do
	w=inportb(0x61) & 0x20;  /*check front from timer out 2  */
	while (w==0);

	do
	w=inportb(0x61) & 0x20;  /*check back front from timer out 2 */
	while (w!=0);

	outportb (0x61,0x40);	/* disable  timer   */

}

/*-------------------------------------------------------------------------*/



/*----------------------------------------------------------------------*/
void program(int id)
{
   int i, j, n, s, w, mask, *buf;

   {
      buf = progbuf;
      s = PSIZE;
      mask = 0x0FFF;
    }
      prog_mode();
      command(INCADD);


   for ( i=0; i<s; ++i )
    { n=0;  /*retry counter */

     do                         /* 25 prog retry max */
      {
      command(LDPROG);
      out_word(buf[i]);
      command(BEGPRG);
      dly100us();
      command(ENDPRG);


      command(RDPROG);
      w = in_word() & mask;
      n++;
      }
      while ( ((buf[i] & mask) != w) && n<=25 );

      if  ((buf[i] & mask) != w)
	{
	 fprintf(stderr,"pp: %04x: %04x != %04x\n",i,w,buf[i]);
	 quit("Verify failed during programming");
	}

       else

	{ for (j=0; j<3*n; j++)       {
					command(LDPROG);
					out_word(buf[i]);
					command(BEGPRG);
					dly100us();
					command(ENDPRG);
				     }
	}


      command(INCADD);
    }




   buf = idbuf;

   if (id==2)    printf("id memory location programming ! \n");

   for ( i=0; i<4; ++i )
   { n=0;  /*retry counter */

     if (id==2)
     {


		do          /* 100 prog retry at max */
		{
		command(LDPROG);
		out_word(idbuf[i]);
		command(BEGPRG);
		dly100us();
		command(ENDPRG);
		command(RDPROG);
		w = in_word() & mask;
		n++;
		}
		while ( (idbuf[i] != w) && n<=100 );


		if  (idbuf[i] != w)
		{
		fprintf(stderr,"pp: %04x: %04x != %04x\n",i,w,idbuf[i]);
		quit(" ID programming failure !");
		}

		else
		{ for (j=0; j<3*n; j++)
					     {	command(LDPROG);
						out_word(idbuf[i]);
						command(BEGPRG);
						dly100us();
						command(ENDPRG); }
		}
     }

     command(INCADD);
   }



}
/*------------------------------------------------------------------------*/

/*------------------------------------------------------------------------*/
void config(void)
{
   int i, j, n, w, mask, *buf;



   printf("Blowing fuses to 0x%02X ...\n",fuses);

   mask=0x1F;
   prog_mode();


    n=0;  /*  */

		do          /* 100 prog retry max */
		{
		command(LDPROG);
		out_word(fuses);
		command(BEGPRG);
		dly100us();
		command(ENDPRG);
		command(RDPROG);
		w = in_word() & mask;
		n++;
		}
		while ( (fuses != w) && n<=100 );

		if ( fuses != w)
		{
		fprintf(stderr,"pp: %02x: != %02x\n",fuses,w);
		quit("Fuse programming failure !");
		}

		else
		{ for (j=0; j<3*n; j++)
					     {	command(LDPROG);
						out_word(fuses);
						command(BEGPRG);
						dly100us();
						command(ENDPRG); }
		}


}
/*------------------------------------------------------------------*/


/*------------------------------------------------------------------------*/
void verify(void)
{
   int i, n, w, mask, rdcmd, *buf;

   {
      buf = progbuf;
      n = PSIZE;
      mask = 0x0FFF;
      rdcmd = RDPROG;
   }

   prog_mode();
   command(INCADD);

   for ( i=0; i<n; ++i ) {
      command(rdcmd);
      w = in_word() & mask;
      if ( (buf[i] & mask) != w ) {
	 fprintf(stderr,"pp: %04x: %04x != %04x\n",i,w,buf[i]);
	 quit("Verify failed");
      }
      command(INCADD);
   }
}
/*------------------------------------------------------------------------*/


int hexdigit(FILE* fp)
{
   int c;

   if ( (c=getc(fp)) == EOF )
      quit("Unexpected EOF");

   c -= c>'9'? 'A'-10: '0';
   if ( c<0 || c>0xF )
      quit("Hex digit expected");
   return c;
}

int hexbyte(FILE* fp)
{
   int b;

   b = hexdigit(fp);
   b = (b<<4) + hexdigit(fp);
   check += b;               /* nasty side-effect, but useful trick */
   return b;
}

unsigned hexword(FILE* fp)
{
   unsigned w;

   w = hexbyte(fp);
   w = (w<<8) + hexbyte(fp);
   return w;
}

#define swab(w) (((w)>>8)+(((w)&0xFF)<<8))   /* swap bytes */

void loadhex(FILE* fp,int* buf,int bufsize)
{
   int type, nwords, i;
   unsigned address, w;

   type = 0;
   while ( type != 1 ) {
      if ( getc(fp) != ':' )
	 quit("Expected ':'");
      check = 0;
      w = hexbyte(fp);
      nwords = (hextype == INHX8M)? w/2: w;
      w = hexword(fp);
      address = (hextype == INHX8M)? w/2: w;
      type = hexbyte(fp);
      for ( i=0; i<nwords; ++i, ++address ) {
	 if ( address >= bufsize )
	    quit("Impossible address");
	 w = hexword(fp);
	 buf[address] = (hextype == INHX8M)? swab(w): w;
      }
      (void) hexbyte(fp);
      (void) getc(fp);
      if ( check&0xFF )
	 quit("Checksum error");
   }
}


/*------------------------------------------------------------------------*/
void blnkprog(void)
{
   int i, n, w, mask, no_abort, in_key, rdcmd, *buf;

      n = PSIZE;
      mask = 0x0FFF;
      no_abort = 0;


   prog_mode();
   command(INCADD);

   for ( i=0; i<n; ++i ) {
      command(RDPROG);
      w = in_word() & mask;
 /*     printf("Adress: %03X = %04X \n",i,w); */

      if ( w != 0X0FFF)
	 {      if (no_abort == 0)
		{
		printf("Device not blank !\n");
		fprintf(stderr,"location : %03x = %04x \nProgramming (y/n)?\n",i,w);
		in_key = getch();
         	if ( (in_key != 'y') && (in_key != 'Y') )
	 	quit("Aborted");
		no_abort = 1;
		}
	 }

      command(INCADD);
   }
}
/*------------------------------------------------------------------------*/


/*------------------------------------------------*/
void blnkfuse(void)
{
   int i,cfg,in_key;

   prog_mode();

   command(RDPROG);
   cfg = (in_word() & 0x1F);

   if ( cfg != 0x1F )
      {
	 printf("Fuses register not blank !\n");
	 fprintf(stderr,"fuse=: %02x\nProgramming (y/n)?\n",cfg);
	 in_key = getch();
         if ( (in_key != 'y') && (in_key != 'Y') )
	 quit("Aborted ");
      }

}
/*------------------------------------------------*/






/*MMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMM*/

void main(int argc,char** argv)
{
   FILE *objfp[2];       /* pointer to files array */
   int i, c, obj_cnt = 0;

   for ( i=1; i<argc; ++i )
   {
      if ( (c = *argv[i]) == '-' || c == '/' )
	 get_option(++argv[i]);
      else if ( obj_cnt < 2 )
	       if ( (objfp[obj_cnt++] = fopen(argv[i],"r")) == NULL )
	       quit("Can't open objfile");
   }

   setup();

   if ( obj_cnt < 1 )
      usage();

   loadhex(objfp[0], progbuf, PSIZE);
   if ( obj_cnt > 1 )
      loadhex(objfp[1], idbuf, IDSIZE);

   printf("%s  %s  %s\n",progname,version,copyright);
   if ( !set_fuses )
      ask_fuses();
   if ( wait ) {
      printf("\nInsert PIC and press a key to start (^C to abort) ...\n");


      printf("\n FUSES: %X",fuses);

      if ( getch() == 3 )
	 quit("Aborted");
   }
   printf("\nProgramming ...\n");

    blnkfuse();     /*  fuse blank check */
    blnkprog();     /*  program blank check    */


   program(obj_cnt);

   printf("Verifying ...\n");
   verify();

   config();

   printf("Finished\n\n");
   idle_mode();
}
