/*
 * ppread.c : A serial mode PIC16C84/71/508/509 reader
 *
 * To read PIC84/71/508/509 ROM contents
 *
 * 12-Jan-97, Ver 0.3 update to read PIC12C508/509
 *
 * This is copyright software, but I do not seek to profit from
 * it.  I wrote it in the hope that it will prove useful.  Feel
 * free to modify the program, but please DO NOT distribute
 * modified versions.  Instead, send me your modifications and
 * improvements so that they may, at my discretion, be included in a
 * future release.
 *
 * The author of this program accepts no responsibility for damages that are
 * caused by this program and make NO WARRANTY or representation, either
 * express or implied, with respect to this software.  This software is
 * provided "AS IS" and you, its user, assume the entire risk when you use it.
 *
 * Although I cannot undertake to support this software, I would
 * welcome bug reports, suggestions or comments which should be
 * sent to:
 *
 *    Steven Willis,
 *    sgw1002@cus.cam.ac.uk
 *
 * Copyright (C) 1997 Steven Willis.
 * This program is free software.  Permission is granted to use,
 * copy, or redistribute this program so long as it is not sold
 * for profit.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <conio.h>

/* #define U7407 */   /* define U7407 if H/W uses non-inverting buffers */

#define IN      64
#define VDD     4
#define VPP     8
#define CLK     2
#define OUT     1

#define LDCONF  0
#define LDPROG  2
#define RDPROG  4
#define INCADD  6
#define BEGPRG  8
#define LDDATA  3
#define RDDATA  5
#define ENDPRG	14

#define VPPDLY  250    /* these are in milliseconds */
#define VDDDLY  250
#define PRGDLY  12

#define TSET    1      /* the units for these are CPU dependent */
#define THLD    1      /* I doubt if they are necessary even    */
#define TDLY    2      /* on the fastest PC, but ...            */
#define TDLY50  1

#define PROGRAM 0
#define DATA    1
#define PSIZE   1024
#define PSIZE508 512
#define DSIZE   64

#define MCLRE   16
#define CP87    16
#define CP50    8
#define PWRTE   8
#define WDTE    4
#define RC      3
#define HS      2
#define XT      1
#define LP      0

#define INHX16  16
#define INHX8M  8

#ifdef U7407
#define inbit           (inportb(s_reg)&IN)
#define vppon           (d_bits &= ~VPP)
#define vppoff          (d_bits |= VPP)
#define vddon           (d_bits &= ~VDD)
#define vddoff          (d_bits |= VDD)
#define clkhi           (d_bits |= CLK)
#define clklo           (d_bits &= ~CLK)
#define outhi           (d_bits |= OUT)
#define outlo           (d_bits &= ~OUT)
#else
#define inbit           (~inportb(s_reg)&IN)
#define vppon           (d_bits |= VPP)
#define vppoff          (d_bits &= ~VPP)
#define vddon           (d_bits |= VDD)
#define vddoff          (d_bits &= ~VDD)
#define clkhi           (d_bits &= ~CLK)
#define clklo           (d_bits |= CLK)
#define outhi           (d_bits &= ~OUT)
#define outlo           (d_bits |= OUT)
#endif /* U7407 */
#define assert          (outportb(d_reg,d_bits))
#define yes     			(c=getche(), c == 'y' || c == 'Y')

int progbuf[PSIZE];
int databuf[DSIZE];
int fuses = 0;

int d_bits = 0;
int d_reg;
int s_reg;
int check;
int iport = 0;
int hextype = INHX16;
int set_fuses = 0;
int wait = 1;
int device = 87;

char *progname = "PIC16C84/71/508/509 ROM reader";
char *version = "Version 0.3";
char *copyright = "Copyright (C) 1997 Steven Willis.";
char *email = "SGW1002@cus.cam.ac.uk";

void idle_mode()
{
   vppoff, clklo, outlo, assert;
   delay(VPPDLY);
   vddoff, assert;
   delay(VDDDLY);
}

void prog_mode()
{
   vppoff, vddon, clklo, outlo, assert;
   delay(VPPDLY);
   vppon, assert;
   delay(VDDDLY);
}

void quit(s)
char *s;
{
   fprintf(stderr,"ppread: %s\n",s);
   idle_mode();
   getch();
   exit(1);
}

void test_hw()
{
    int b;
	 vddon, outhi, assert;   /* better have VDD on in case PIC is there */
	 delay(VPPDLY);
	 b = inbit;
	 outlo, assert;
	 delay(TSET);
	 if ( b != IN || inbit != 0 ) {
		 fprintf(stderr,"ppread: Using LPT at 0x%04X\n",d_reg);
		 quit("Hardware fault.  Check power, connections and port used.");
	 }
	 idle_mode();
}

void setup()
{
   int i, b;

   vppoff, vddoff, clklo, outlo, assert;
   d_reg = peek(0,0x408+iport);  /* find the base address of LPT port */
   s_reg = d_reg+1;

   switch ( d_reg ) {  /* I don't want to blow anything up, so check port */
       case 0x3BC:
       case 0x378:
       case 0x278: break;
       default:  fprintf(stderr,"ppread: LPT at 0x%04X\n",d_reg);
		 quit("Unlikely LPT address");
   }

   test_hw();

   for ( i=0; i<PSIZE; ++i )
      progbuf[i] = 0x3FFF;
   for ( i=0; i<DSIZE; ++i )
      databuf[i] = 0xFF;
}

void tiny_delay(ticks)
int ticks;
{
		while ( ticks-- );
}

void clock_out(bit)
int bit;
{
   bit? outhi: outlo, clkhi, assert;
   tiny_delay(TSET);
   clklo, assert;
   tiny_delay(THLD);
   outlo, assert;
}

int clock_in()
{
   int b;

   outhi, clkhi, assert;
   tiny_delay(TSET);
   clklo, assert;
   b = inbit? 1: 0;
   tiny_delay(THLD);
   return b;
}

void out_word(w)
int w;
{
   int b;

   clock_out(0);
   for ( b=0; b<14; ++b )
     clock_out(w&(1<<b));
   clock_out(0);
}

int in_word()
{
   int b, w;

   (void) clock_in();
   for ( w=0, b=0; b<14; ++b )
     w += clock_in()<<b;
   (void) clock_in();
   return w;
}

void command(cmd)
int cmd;
{
   int b;

   outlo, assert;
	if(device == 87) tiny_delay(TDLY);
	else delay(TDLY50);
	for ( b=0; b<6; ++b )
		clock_out(cmd&(1<<b));
	outhi, assert;
	if(device == 87) tiny_delay(TDLY);
	else delay(TDLY50);
}

void read()
{
   int i=0, n, mask;
	if(device == 87)  { n = PSIZE;    mask = 0x3FFF; }
	if(device == 508) { n = PSIZE508; mask = 0x0FFF; }
	if(device == 509) { n = PSIZE;    mask = 0x0FFF; }
	prog_mode();
	if(device == 508 || device == 509)
	{
		command(RDPROG);
		fuses = in_word() & mask;
		command(INCADD);
	}
	for(i=0; i<n; i++) {
      command(RDPROG);
      progbuf[i] = in_word() & mask;
      command(INCADD);
   }
   idle_mode();
}

void display()
{
   int i=0, l, p=0, n, mask, key;
	if(device == 87)  { n = PSIZE;    mask = 0x3FFF; }
	if(device == 508) { n = PSIZE508; mask = 0x0FFF; }
	if(device == 509) { n = PSIZE;    mask = 0x0FFF; }
	printf("Address        0      1      2      3      4      5      6      7\n");
   do{
      printf("\n   %4X     ",i);
      l=0;
      do{
			printf("%4X   ", (progbuf[i] & mask));
			l++;
			i++;
		}while(l<8);
		p++;
		if(p==16){
			printf("\n");
			p=0;
			key=getch();
      }
		if(key=='x' || key=='X') {
			if(device == 87 || device == 509) i=PSIZE;
			else i=PSIZE508;
		}
   }while(i<n);
}

void config()
{
   int i;
	if(device == 87) {
		prog_mode();
		command(LDCONF);
		out_word(fuses);
		for ( i=0; i<7; ++i )
		command(INCADD);
		command(RDPROG);
		fuses = (in_word()&0x1F);
	}
	printf("\nFuses are set to: ");
	if(device == 87)
	{
		if(fuses & CP87) printf("CP off, ");
		else printf("CP on, ");
		if(fuses & PWRTE) printf("PWRTE on, ");
		else printf("PWRTE off, ");
	}
	else
	{
		if(fuses & MCLRE) printf("MCLRE on, ");
		else printf("MCLRE off, ");
		if(fuses & CP50) printf("CP off, ");
		else printf("CP on, ");
	}
	if(fuses & WDTE) printf("WDTE on, ");
   else printf("WDT off, ");
   fuses = fuses & 0x03;
   if(fuses == RC) printf("RC osc\n");
   if(fuses == HS) printf("HS osc\n");
   if(fuses == XT) printf("XT osc\n");
   if(fuses == LP) printf("LP osc\n");
   idle_mode();
}

void main()
{
	int i, c, key;
	clrscr();
	setup();
	printf("%s  %s  %s\n",progname,version,copyright);
	do{
		printf("\nPlease enter device type (1-84/71, 2-508, 3-509) ? ");
		key = getche();
	}while(key<'1' || key>'3');
	if(key=='1') {device=87; printf("\nReading PIC16C84/71.. (X - Exit)\n\n");}
	if(key=='2') {device=508; printf("\nReading PIC12C508... (X - Exit)\n\n");}
	if(key=='3') {device=509; printf("\nReading PIC12C509... (X - Exit)\n\n");}
	read();
	display();
   config();
   printf("\nFinished ....\n\n");
   idle_mode();
}
